"use strict";

/**
 * Module dependencies.
 */

const util = require('util');
const Stream = require('stream');
const ResponseBase = require('../response-base');
const _require = require('../utils'),
  mixin = _require.mixin;

/**
 * Expose `Response`.
 */

module.exports = Response;

/**
 * Initialize a new `Response` with the given `xhr`.
 *
 *  - set flags (.ok, .error, etc)
 *  - parse header
 *
 * @param {Request} req
 * @param {Object} options
 * @constructor
 * @extends {Stream}
 * @implements {ReadableStream}
 * @api private
 */

function Response(request) {
  Stream.call(this);
  this.res = request.res;
  const res = this.res;
  this.request = request;
  this.req = request.req;
  this.text = res.text;
  this.files = res.files || {};
  this.buffered = request._resBuffered;
  this.headers = res.headers;
  this.header = this.headers;
  this._setStatusProperties(res.statusCode);
  this._setHeaderProperties(this.header);
  this.setEncoding = res.setEncoding.bind(res);
  res.on('data', this.emit.bind(this, 'data'));
  res.on('end', this.emit.bind(this, 'end'));
  res.on('close', this.emit.bind(this, 'close'));
  res.on('error', this.emit.bind(this, 'error'));
}

// Lazy access res.body.
// https://github.com/nodejs/node/pull/39520#issuecomment-889697136
Object.defineProperty(Response.prototype, 'body', {
  get() {
    return this._body !== undefined ? this._body : this.res.body !== undefined ? this.res.body : {};
  },
  set(value) {
    this._body = value;
  }
});

/**
 * Inherit from `Stream`.
 */

util.inherits(Response, Stream);
mixin(Response.prototype, ResponseBase.prototype);

/**
 * Implements methods of a `ReadableStream`
 */

Response.prototype.destroy = function (error) {
  this.res.destroy(error);
};

/**
 * Pause.
 */

Response.prototype.pause = function () {
  this.res.pause();
};

/**
 * Resume.
 */

Response.prototype.resume = function () {
  this.res.resume();
};

/**
 * Return an `Error` representative of this response.
 *
 * @return {Error}
 * @api public
 */

Response.prototype.toError = function () {
  const req = this.req;
  const method = req.method;
  const path = req.path;
  const message = `cannot ${method} ${path} (${this.status})`;
  const error = new Error(message);
  error.status = this.status;
  error.text = this.text;
  error.method = method;
  error.path = path;
  return error;
};
Response.prototype.setStatusProperties = function (status) {
  console.warn('In superagent 2.x setStatusProperties is a private method');
  return this._setStatusProperties(status);
};

/**
 * To json.
 *
 * @return {Object}
 * @api public
 */

Response.prototype.toJSON = function () {
  return {
    req: this.request.toJSON(),
    header: this.header,
    status: this.status,
    text: this.text
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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